#include <stdarg.h>
#include <linux/proc_fs.h>
#include <linux/mm.h>
#include <asm/system.h>
#include <asm/delay.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <mach/intctl-regs.h>
#include <asm/mach/map.h>
#include <dtvrecdd/memtrans.h>

#include "iprq.h"
#include "iprq_common.h"

unsigned long iprq_gxicr_base0;
unsigned long iprq_gxicr_base1;
unsigned long iprq_cpuint_base;
unsigned long iprq_mem_addr;

int __iprq_initialized = 0;

void entry_iprq(void);

static irqreturn_t proc_request_interrupt(int irq, void *dev_id);

static irqreturn_t proc_request_interrupt(int irq, void *dev_id)
{
#ifdef IPRQ_DEBUG
	printk("proc_reqeust_interrupt! irq=%d\n", irq);
#endif
	CLR_CPUINT(CPUINT_FROM_OS_R);

	do_proc_request();

	return IRQ_HANDLED;
}

#ifdef IPRQ_POLLING_ENABLE

static struct timer_list iprq_timer;

static void iprq_timer_request(unsigned long data)
{
	del_timer(&iprq_timer);
	do_proc_request();
	iprq_timer.expires = jiffies + 1;
	add_timer(&iprq_timer);
}
#endif /* IPRQ_POLLING_ENABLE */

#define is_iprq_booted(osid)	IS_IRQ_ENABLE(GET_IPRQ_IRQ(osid))

static int __iprq_wait_for_sync(void)
{
	/* Wait until iprq initialization */
	while (!is_iprq_booted(OS_R));

	__iprq_initialized = 1;
	return 0;
}

static int iprq_read_proc_dump(char *, char **, off_t, int, int *, void *);

int __init init_iprq(void)
{
	int irq;
	int ret;
	struct proc_dir_entry *ent, *dir;
	struct iprq_status_t *status;
	unsigned long flags;

	iprq_gxicr_base0 = (unsigned long)ioremap_nocache(__GXICR_BASE0, 0x1000);
	if (!iprq_gxicr_base0)
		return -1;
	iprq_gxicr_base1 = (unsigned long)ioremap_nocache(__GXICR_BASE1, 0x1000);
	if (!iprq_gxicr_base1)
		return -1;
	iprq_cpuint_base = (unsigned long)ioremap_nocache(__CPUINT_BASE, 0x1000);
	if (!iprq_cpuint_base)
		return -1;

#ifdef IPRQ_L2_CACHEABLE
	iprq_mem_addr = (unsigned long)__arch_ioremap(IPRQ_MEM_PADDR, IPRQ_MEM_SIZE, MT_MEMORY_UNCACHED_WB);
#else
	iprq_mem_addr = (unsigned long)ioremap_nocache(IPRQ_MEM_PADDR, IPRQ_MEM_SIZE);
#endif

	dir = proc_mkdir("iprq", NULL);
	if (dir) {
		ent = create_proc_entry("dump", S_IFREG | S_IRUGO, dir);
		if (ent) {
			ent->read_proc  = iprq_read_proc_dump;
		}
	}
#ifdef IPRQ_DEBUG
	printk("IPRQ Map: %08x -> %08lx +%08x\n", IPRQ_MEM_PADDR, iprq_mem_addr, IPRQ_MEM_SIZE);
#endif
	__iprq_wait_for_sync();

	entry_iprq();

	irq = GET_IPRQ_ARMGIC(SELF_OSID());
	ret = request_irq(irq, proc_request_interrupt, 0,
			  "inter processor request intr", NULL);
	if (ret < 0) {
		printk("init_iprq: request_irq error:%d\n", ret);
		return ret;
	}

#ifdef IPRQ_POLLING_ENABLE
	init_timer(&iprq_timer);
	iprq_timer.function = iprq_timer_request;
	iprq_timer.data = 0; /* unused */
	iprq_timer.expires = jiffies + 1;
	add_timer(&iprq_timer);
#endif /* IPRQ_POLLING_ENABLE */

	status = iprq_status;
	flags = LOCK_CLI_SAVE(&status->lock);
	__set_os_initialized(status, SELF_OSID());
	UNLOCK_IRQ_RESTORE(&status->lock, flags);

	return 0;
}

void __exit exit_iprq(void)
{
        int irq = GET_IPRQ_ARMGIC(SELF_OSID());
        int i;

#ifdef IPRQ_POLLING_ENABLE
        del_timer_sync(&iprq_timer);
#endif
        free_irq(irq, NULL);

        for (i = 0; i < IPRQ_REQUEST_MAXNUM; i++) {
                iprqL_entry(i, NULL);
        }

        iounmap((void *)iprq_mem_addr);
        iounmap((void *)iprq_gxicr_base0);
        iounmap((void *)iprq_gxicr_base1);
        iounmap((void *)iprq_cpuint_base);
}

//module_init(init_iprq);
//module_exit(exit_iprq);


#define MAX_CHARS_PER_LINE	64

static int __iprq_dump_print(char *buf, char *tmp_buf,
				int count, off_t off, int pos,
				int fix_len, const char *format, ...)
{
	int len;
	va_list args;

	va_start(args, format);

	if (fix_len > 0 && off >= pos + fix_len) {
		return pos + fix_len;
	}

	if (off > pos) {
		len = vsnprintf(tmp_buf, MAX_CHARS_PER_LINE, format, args);
		va_end(args);
		if (off >= pos + len) {
			return pos + len;
		}
		if (pos + len > off + count) {
			len = off + count - pos;
		}
		memcpy(buf, tmp_buf + off - pos, len);
		return pos + len;
	}

	if (off + count >= pos + MAX_CHARS_PER_LINE) {
		len = vsnprintf(buf + pos - off, MAX_CHARS_PER_LINE, format, args);
		va_end(args);
		return pos + len;
	}

	len = vsnprintf(tmp_buf, MAX_CHARS_PER_LINE, format, args);
	va_end(args);
	if (pos + len > off + count) {
		len = off + count - pos;
	}
	memcpy(buf + pos - off, tmp_buf, len);
	return pos + len;
}

#define DUMPMEM_CHARS_OF_ADDR_INFO	12
#define DUMPMEM_CHARS_PER_1BYTE_DATA	3
#define DUMPMEM_CHARS_PER_LINE		(DUMPMEM_CHARS_OF_ADDR_INFO \
					+ (DUMPMEM_CHARS_PER_1BYTE_DATA * 16))

static int __iprq_dump_memory_range(unsigned long base, unsigned long size,
				int is_physaddr, unsigned int mtype,
				char *buf, char *tmp_buf,
				int count, off_t off, int pos)
{
	unsigned long index;
	char *ptr = NULL;

	if (is_physaddr) {
		ptr = __arch_ioremap(base, size, mtype);
		if (!ptr) {
			printk(KERN_ERR "IPRQ: %s: ioremap error.\n", __func__);
			return -ENOMEM;
		}
	}
	else {
		ptr = (char *)base;
	}

	for (index = 0; index + 16 < size; index += 16) {
		if (pos + DUMPMEM_CHARS_PER_LINE < off) {
			pos += DUMPMEM_CHARS_PER_LINE;
			continue;
		}
		pos = __iprq_dump_print(buf, tmp_buf,
					count, off, pos,
					DUMPMEM_CHARS_OF_ADDR_INFO,
					"\n0x%08lX:", base + index);
		if (pos >= off + count) {
			goto finish;
		}
		pos = __iprq_dump_print(buf, tmp_buf,
				count, off, pos,
				DUMPMEM_CHARS_PER_1BYTE_DATA * 16,
				" %02X %02X %02X %02X %02X %02X %02X %02X"
				" %02X %02X %02X %02X %02X %02X %02X %02X",
				 *((unsigned char *)(ptr + index    )),
				 *((unsigned char *)(ptr + index + 1)),
				 *((unsigned char *)(ptr + index + 2)),
				 *((unsigned char *)(ptr + index + 3)),
				 *((unsigned char *)(ptr + index + 4)),
				 *((unsigned char *)(ptr + index + 5)),
				 *((unsigned char *)(ptr + index + 6)),
				 *((unsigned char *)(ptr + index + 7)),
				 *((unsigned char *)(ptr + index + 8)),
				 *((unsigned char *)(ptr + index + 9)),
				 *((unsigned char *)(ptr + index + 10)),
				 *((unsigned char *)(ptr + index + 11)),
				 *((unsigned char *)(ptr + index + 12)),
				 *((unsigned char *)(ptr + index + 13)),
				 *((unsigned char *)(ptr + index + 14)),
				 *((unsigned char *)(ptr + index + 15)));
		if (pos >= off + count) {
			goto finish;
		}
	}
	while (index  < size) {
		if (!(index & 0xf)) {
			pos = __iprq_dump_print(buf, tmp_buf,
						count, off, pos,
						DUMPMEM_CHARS_OF_ADDR_INFO,
						"\n0x%08lX:", base + index);
			if (pos >= off + count) {
				goto finish;
			}
		}
		pos = __iprq_dump_print(buf, tmp_buf,
					count, off, pos,
					DUMPMEM_CHARS_PER_1BYTE_DATA,
					" %02X",
					*((unsigned char *)(ptr + index++)));
		if (pos >= off + count) {
			goto finish;
		}
	}
	pos = __iprq_dump_print(buf, tmp_buf, count, off, pos, 1, "\n");

finish:
	if (is_physaddr) {
		iounmap(ptr);
	}
	return pos;
}

static int count_before_dump_dsp_reg = 0;
static int count_before_dump_dsp_mem = 0;
static int count_before_dump_iprq_mem = 0;
static int count_after_dump_all = 0;

static int iprq_read_proc_dump(char *page, char **start, off_t off, int count,
				int *eof, void *data)
{
	int pos = 0;
	char *tmp_buf;

	tmp_buf = kmalloc(MAX_CHARS_PER_LINE + 1, GFP_KERNEL);
	if (!tmp_buf) {
		printk(KERN_ERR "IPRQ: %s: cannot allocate buffer.\n", __func__);
		return -ENOMEM;
	}

	if (count_after_dump_all && off >= count_after_dump_all) {
		pos = count_after_dump_all;
		*eof = 1;
		goto finish;
	}
	else if (count_before_dump_iprq_mem && off >= count_before_dump_iprq_mem) {
		pos = count_before_dump_iprq_mem;
		goto dump_iprq_mem;
	}
	else if (count_before_dump_dsp_mem && off >= count_before_dump_dsp_mem) {
		pos = count_before_dump_dsp_mem;
		goto dump_dsp_mem;
	}
	else if (count_before_dump_dsp_reg && off >= count_before_dump_dsp_reg) {
		pos = count_before_dump_dsp_reg;
		goto dump_dsp_reg;
	}

	/* Dump DSP registers */
	pos = __iprq_dump_print(page, tmp_buf, count, off, pos, 0,
			"IPRQ: dump DSP registers (phys addr:0x%08lX~0x%08lX)\n",
			DDR_DSP_STATUS_BASE,
			DDR_DSP_STATUS_BASE + DDR_DSP_STATUS_SIZE - 1);
	if (pos >= off + count) {
		goto finish;
	}

	count_before_dump_dsp_reg = pos;

dump_dsp_reg:
	pos = __iprq_dump_memory_range(DDR_DSP_STATUS_BASE, DDR_DSP_STATUS_SIZE,
				1, MT_DEVICE,
				page, tmp_buf,
				count, off, pos);
	if (pos < 0 || pos >= off + count) {
		goto finish;
	}

	/* Dump DSP memory */
	pos = __iprq_dump_print(page, tmp_buf, count, off, pos, 0,
			"\nIPRQ: dump DSP memory (phys addr:0x%08lX~0x%08lX)\n",
			DDR_CH0_START + DDR_CH0_OFFSET + DDR_IPP_ITRON_OFFSET,
			DDR_CH0_START + DDR_CH0_OFFSET + DDR_IPP_ITRON_OFFSET
			 + DDR_IPP_ITRON_SIZE - 1);
	if (pos >= off + count) {
		goto finish;
	}

	count_before_dump_dsp_mem = pos;

dump_dsp_mem:
	pos = __iprq_dump_memory_range(
				DDR_CH0_START + DDR_CH0_OFFSET + DDR_IPP_ITRON_OFFSET,
				DDR_IPP_ITRON_SIZE, 1, MT_MEMORY_UNCACHED_WB,
				page, tmp_buf,
				count, off, pos);
	if (pos < 0 || pos >= off + count) {
		goto finish;
	}

	/* Dump IPRQ memory */
	pos = __iprq_dump_print(page, tmp_buf, count, off, pos, 0,
			"\nIPRQ: dump IPRQ memory (virt addr:0x%08lX~0x%08lX)\n",
			iprq_mem_addr, iprq_mem_addr + IPRQ_MEM_SIZE - 1);
	if (pos >= off + count) {
		goto finish;
	}

	count_before_dump_iprq_mem = pos;

dump_iprq_mem:
	pos = __iprq_dump_memory_range(iprq_mem_addr, IPRQ_MEM_SIZE, 0, 0,
				page, tmp_buf,
				count, off, pos);
	if (pos < 0 || pos >= count + off) {
		goto finish;
	}
	count_after_dump_all = pos;
	*eof = 1;

finish:
	kfree(tmp_buf);

	if (pos < 0) {
		return pos;
	}

	if (pos > off) {
		*start = (char *)(pos - off);
	}
	return (pos > off) ? pos - off : 0;
}
